package jp.ac.osaka_u.ist.sel.android.simn.impl;

import jp.ac.osaka_u.ist.sel.android.simn.BlockDirection;
import jp.ac.osaka_u.ist.sel.android.simn.BlockType;
import jp.ac.osaka_u.ist.sel.android.simn.Horizon;
import jp.ac.osaka_u.ist.sel.android.simn.Vertical;
import android.util.Log;

public class BlockOperator implements IntervalEventListener
{
  /* inner class */
  private interface Progress
  {
    Point getNextPoint(Point p);
    Point getPreviousPoint(Point p);
  }

  private class Forward implements Progress
  {
    public Point getNextPoint(Point p)
    {
      return new Point(p.getX(), p.getY()+1);
    }

    public Point getPreviousPoint(Point p)
    {
      return new Point(p.getX(), p.getY()-1);
    }
  }

  private class Rightward implements Progress
  {
    public Point getNextPoint(Point p)
    {
      return new Point(p.getX()+1, p.getY());
    }
    
    public Point getPreviousPoint(Point p)
    {
      return new Point(p.getX()-1, p.getY());
    }
  }

  interface PointDelegate {
    Point getPosition(Point p);
  }

  /* static methods */
  static void exchange(BlockOperator b1, BlockOperator b2)
  {
    {
      Block tmp = b1.block_;
      b1.block_ = b2.block_;
      b2.block_ = tmp;
    }
    {
      int tmp = b1.rightTurnNum_;
      b1.rightTurnNum_ = b2.rightTurnNum_;
      b2.rightTurnNum_ = tmp;
    }
    {
      RealPoint tmp = b1.centerPos_;
      b1.centerPos_ = b2.centerPos_;
      b2.centerPos_ = tmp;
    }
    {
      FuturePosition tmp = b1.futurePos_;
      b1.futurePos_ = b2.futurePos_;
      b2.futurePos_ = tmp;
    }
    {
    	BlockType tmp = b1.blockType_;
        b1.blockType_ = b2.blockType_;
        b2.blockType_ = tmp;
      }
  }
  
  /* read only varibales */
  private final Delayed remainingDelayed_;
  private final Delayed totalDelayed_;
  private final FieldOperator field_;
  private BlockType blockType_;
  private final PitchHolder pitch_;
  private final Progress vertical_;
  private final Progress horizon_;
  
  /* private variables */
  private Block block_;
  private int rightTurnNum_;
  private RealPoint centerPos_;
  private FuturePosition futurePos_;
  
  /* constructor */
  public BlockOperator(BlockType t, FieldOperator field, Point centerPos, Delayed totalDelayed,
                       Delayed remainingDelayed, PitchHolder pitch) throws NotCreateBlock
  {
    block_ =  new Block(t, centerPos);
    vertical_ = new Forward();
    horizon_ = new Rightward();
    field_ = field;
    rightTurnNum_ = Integer.MAX_VALUE/2;
    rightTurnNum_ -= rightTurnNum_ % BlockDirection.values().length;
    blockType_ = t;
    Point p = block_.getCenterCell().getPosition();
    centerPos_ = new RealPoint(p.getX(), p.getY());
    totalDelayed_ = totalDelayed;
    remainingDelayed_ = remainingDelayed;
    pitch_ = pitch;
    futurePos_ = new FuturePosition();
    updateFuturePosition(futurePos_, block_, field_);
    
    if (!validBlock(block_, field_))
      throw new NotCreateBlock();
  }

  /* private methods */
  private boolean validBlock(Block block, FieldOperator field)
  {
    for (Cell c : block.getCells())
      if (!field.isEmpty(c.getPosition()))
        return false;
    return true;
  }
  
  private Point nextRelativePosition(Point p, double angle)
  {
    double x = p.getX();
    double y = p.getY();
    double len = Math.sqrt(x*x+y*y);
    double normal_x = x / len;
    double normal_y = y / len;
    double result_angle = angle + Math.atan(normal_y/normal_x) + ((normal_x >= 0.0) ? 0.0 : Math.PI);
    return new Point((int)Math.round(Math.cos(result_angle)*len), (int)Math.round(Math.sin(result_angle)*len));
  }

  private Point toObsoletePosition(Block block, Point relativePos)
  {
    return Point.add(block.getCenterCell().getPosition(), relativePos);
  }

  private Point toRelativePosition(Block block, Cell c)
  {
    return toRelativePosition(block, c.getPosition());
  }

  private Point toRelativePosition(Block block, Point p)
  {
    return Block.toRelativePosition(block.getCenterCell().getPosition(), p);
  }

  private boolean move(PointDelegate d)
  {
    Block block = new Block();

    for (Cell c : block_.getCells())  {
        Point next = d.getPosition(c.getPosition());
        /* success */
        if (field_.isEmpty(next))
          block.setCell(new Cell(next, blockType_),
                        Block.toPositionWithinBlockFromRelativePosition(toRelativePosition(block_, c)));
        /* fail */
        else
          return false;
      }

    block_ = block;
    return true;
  }  

  /*
  private boolean canMove(PointDelegate d)
  {
    for (Cell c : this.getCells())
      if (!field_.isEmpty(d.getPosition(c.getPosition())))
        return false;
    return true;
  }
  */
  
  private PointDelegate createMoveDirect(Vertical direction)
  {
    if (direction == Vertical.Down)
      return new PointDelegate() { public Point getPosition(Point p) {
        return vertical_.getNextPoint(p);
      }};
    else
      return new PointDelegate() { public Point getPosition(Point p) {
        return vertical_.getPreviousPoint(p);
      }};
  }

  public void updateFuturePosition(FuturePosition futurePos, Block block, FieldOperator field)
  {
    futurePos.updatePosition(block, field);
  }

  /* default methods */
  boolean canChangeCenterPosition(Point p)
  {
    synchronized (this) {
      for (Cell c : block_.getCells()) {
        if (!field_.isEmpty(Point.add(toRelativePosition(block_, c), p)))
          return false;
      }
      return true;
    }
  }
  
  void setCenterPosition(Point p)
  {
    assert canChangeCenterPosition(p);
    synchronized (this) {
      Block block = new Block();

      for (Cell c : block_.getCells()) {
        Point relativePos = toRelativePosition(block_, c);
        block.setCell(new Cell(Point.add(relativePos, p), blockType_), 
                      Block.toPositionWithinBlockFromRelativePosition(relativePos));
      }
      block_ = block;
    }
  }

  void update()
  {
    synchronized (this) {
      updateFuturePosition(futurePos_, block_, field_);
    }
  }

  boolean withinBlock(int height_index, int width_index)
  {
    return this.withinBlock(new Point(width_index, height_index));
  }
  
  boolean withinBlock(Point p)
  {
    p = Block.toPositionWithinBlockFromRelativePosition(this.toRelativePosition(block_, p));
    return
      (0 <= p.getY() && p.getY() < block_.sizeLine()) &&
      (0 <= p.getX() && p.getX() < block_.sizeColumn()) &&
      (block_.getCell(p.getY(), p.getX()) != null);
  }
  
  /* public methods */
  public Point getCenterCellPosition()
  {
    return block_.getCenterCell().getPosition();
  }
  
  public void turn(Horizon direction)
  {
    synchronized (this) {
      double rotate_angle = (direction == Horizon.Right) ? Math.PI/2 : -Math.PI/2;
      Block block = new Block();

      for (Cell c : block_.getCells()) {
        Point nextPos = nextRelativePosition(toRelativePosition(block_, c), rotate_angle);
        Point obsoletePos = toObsoletePosition(block_, nextPos);
        /* success */
        if (field_.isEmpty(obsoletePos))
          block.setCell(new Cell(obsoletePos, blockType_), 
                        Block.toPositionWithinBlockFromRelativePosition(nextPos));
        /* fail */
        else
          return ;
      }
      
      block_ = block;rightTurnNum_ += (direction == Horizon.Right) ? 1 : -1;
      updateFuturePosition(futurePos_, block_, field_);
    }
  }

  public void move(Horizon direction)
  {
    synchronized (this) {
      PointDelegate d;
      if (direction == Horizon.Right)
        d = new PointDelegate() { public Point getPosition(Point p) {
          return horizon_.getNextPoint(p);
        }};
      else
        d = new PointDelegate() { public Point getPosition(Point p) {
          return horizon_.getPreviousPoint(p);
        }};

      move(d);
      centerPos_ = new RealPoint(block_.getCenterCell().getPosition().getX(),
                                 centerPos_.getY());
      updateFuturePosition(futurePos_, block_, field_);
    }
  }
  
  public boolean move(Vertical direction)
  {
    synchronized (this) {
      try {
        boolean ret = move(createMoveDirect(direction));
        centerPos_ = new RealPoint(centerPos_.getX(),
                                   block_.getCenterCell().getPosition().getY());
        return ret;
      } catch (Exception ex) {
        Log.d("impl", ex.toString());
        return false;
      }
    }
  }

  public Iterable<Cell> getCells()
  {
    synchronized (this) {
      return block_.getCells();
    }
  }
  
  public BlockDirection getDirection()
  {
    switch (rightTurnNum_ % BlockDirection.values().length) {
    case 0:
      return BlockDirection.Front;
    case 1:
      return BlockDirection.Right;
    case 2:
      return BlockDirection.Back;
    case 3:
      return BlockDirection.Left;
    default:
      assert false : "unknown turn!";
      return BlockDirection.Back; // return dummy value
    }
  }
  
  public BlockType getBlockType()
  {
	  return blockType_; 
  }
  
  public RealPoint getCenterPosition()
  {
    return centerPos_;
  }

  public Point getFutureCenterPosition()
  {
    return futurePos_.getCenterPosition(pitch_.getMoveDirection());
  }
  
  /* for interface IntervalEventListener */
  public void action()
  {
    /*
    Vertical dir = pitch_.getMoveDirection();
    RealPoint p = new RealPoint(0.0, 0.0);
    if (canMove(createMoveDirect(pitch_.getMoveDirection())))
      p = new RealPoint(0.0,
                        (totalDelayed_.getDelay()-remainingDelayed_.getDelay()) / (double)totalDelayed_.getDelay());

    if (dir == Vertical.Up) {
      try {
        for (Cell c : this.getCells())
          if (!field_.isEmpty(new Point((int)Math.round(c.getPosition().getX()),
                                        (int)Math.round(Math.ceil(c.getPosition().getY()+p.getY())))))
            throw new Exception();
      } catch (Exception e) {
        p = new RealPoint(0.0, 0.0);
      }
    }
    */
    
    //Vertical dir = pitch_.getMoveDirection();
    RealPoint p = new RealPoint(0.0, 0.0);
    p = new RealPoint(0.0,
                      (totalDelayed_.getDelay()-remainingDelayed_.getDelay()) / (double)totalDelayed_.getDelay());
    try {
      for (Cell c : this.getCells())
        if (!field_.isEmpty(new Point((int)Math.round(c.getPosition().getX()),
                                      (int)Math.round(Math.ceil(c.getPosition().getY()+p.getY())))))
          throw new DummyThrowable();
    } catch (DummyThrowable e) {
      p = new RealPoint(0.0, 0.0);
    }
    /*
    try {
      switch (dir) {
      case Up:
        for (Cell c : this.getCells())
          if (!field_.isEmpty(new Point((int)Math.round(c.getPosition().getX()),
                                        (int)Math.round(Math.ceil(c.getPosition().getY()+p.getY())))))
            throw new Exception();
        break;
      case Down:
      for (Cell c : this.getCells())
          if (!field_.isEmpty(new Point((int)Math.round(c.getPosition().getX()),
                                        (int)Math.round(Math.ceil(c.getPosition().getY()+p.getY())))))
            throw new Exception();

      }
    } catch (Exception e) {
      p = new RealPoint(0.0, 0.0);
    }
    */
    centerPos_ = RealPoint.add(p, block_.getCenterCell().getPosition());
    //System.out.println("y:" + centerPos_.getY() + ", total:" + totalDelayed_.getDelay() + ", remaining:" + remainingDelayed_.getDelay() );
    //android.util.Log.d("CenterPos", "y:" + Double.toString(centerPos_.getY()) + ", total:" + Long.toString(totalDelayed_.getDelay()) + ", remaining:" + Long.toString(remainingDelayed_.getDelay() ));

  }
  
}
