package jp.ac.osaka_u.ist.sel.android.simn.impl;

import java.util.Arrays;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.concurrent.Callable;

import jp.ac.osaka_u.ist.sel.android.simn.BlockType;
import jp.ac.osaka_u.ist.sel.android.simn.CutCompleteListener;
import jp.ac.osaka_u.ist.sel.android.simn.EventNotifier;
import jp.ac.osaka_u.ist.sel.android.simn.FallEventListener;
import jp.ac.osaka_u.ist.sel.android.simn.GameSetting;
import jp.ac.osaka_u.ist.sel.android.simn.MoveLineEventListener;
import jp.ac.osaka_u.ist.sel.android.simn.PitchListener;
import jp.ac.osaka_u.ist.sel.android.simn.RotateEventListener;
import jp.ac.osaka_u.ist.sel.android.simn.ShakeListener;


public class Game implements GameListener, BlockCreator, CutListener
{
  static final int NEXT_BLOCKS_NUM = 2;
  static final Point CENTER_POINT = new Point(4, 0);
  static final int INIT_WAIT_TIME = 1000;
  static final int INIT_INTERVAL = 1000;
  static final int BASE_SCORE = 100;
  static final int BONUS_SCORE = 10;
  
  private ArrayList<GameListener> gameListeners_;
  private FieldOperator field_;
  private BlockOperator current_block_;
  private LinkedList<BlockType> next_blocks_;
  private ArrayList<BlockCreationListener> creationListeners_; 

  private LineMoveReceiver moveReceiver_;
  private TurnReceiver turnReceiver_;
  private PitchHolder pitch_;
  private CutCompleteReceiver cutReceiver_;
  private ShakeReceiver shakeReceiver_;
  
  private AutomaticProcessThread processThread_;
  private ThreadTask remainingTask_;
  private ArrayList<BlockDownEventListener> blockDownListeners_;
  private int score_;
  private DelayedHolder totalDelayed_;
  private DelayedHolder remainingDelayed_;
  private ArrayList<IntervalEventListener> intervals_;
  private BlockGenerator blockGenerator_;
  private HoldBlock holdBlock_;
  private ShakePossibility shakePossibility_;
  
  public Game()
  {
    gameListeners_ = new ArrayList<GameListener>();
    this.addGameListener(this);
    creationListeners_ = new ArrayList<BlockCreationListener>();
    current_block_ = null;
    next_blocks_ = null;
    blockDownListeners_ = new ArrayList<BlockDownEventListener>();
    blockGenerator_ = new BlockGenerator();
    field_ = new FieldOperator();
    field_.addCutListener(this);
    next_blocks_ = new LinkedList<BlockType>();
    for (int i = 0; i < NEXT_BLOCKS_NUM; ++i)
      next_blocks_.add(blockGenerator_.getNext());
    moveReceiver_ = new LineMoveReceiver();
    turnReceiver_ = new TurnReceiver();
    pitch_ = new PitchHolder();
    cutReceiver_ = new CutCompleteReceiver();
    shakeReceiver_ = new ShakeReceiver(field_,
                                       new Runnable() { public void run() { stop(); } },
                                       new Runnable() { public void run() { restart(); } },
                                       new Callable<Boolean>() { public Boolean call() { return field_.canShake() && shakePossibility_.canShake(); } },
                                       new Runnable() { public void run() { shakePossibility_.shaked(); } });
                                       
    score_ = 0;
    shakePossibility_ = new ShakePossibility();
    totalDelayed_ = new DelayedHolder();
    remainingDelayed_ = new DelayedHolder();
    intervals_ = new ArrayList<IntervalEventListener>();
    holdBlock_ = new HoldBlock(new BlockCreator() {
                                 public ThreadTask createNewBlock() throws NotCreateBlock
                                 {
                                   stop();
                                   try {
                                     remainingTask_ = startNextAction();
                                   } catch (NotCreateBlock e) {
                                     for (GameListener l : gameListeners_)
                                       l.losed();
                                     remainingTask_ = new ThreadTask(new DummyTask(), Long.MAX_VALUE);
                                   }
                                   restart();
                                   return remainingTask_;
                                 }
                               });
  }

  /* private methods */
  private ThreadTask startNextAction() throws NotCreateBlock
  {
    next_blocks_.add(blockGenerator_.getNext());
    current_block_ = new BlockOperator(next_blocks_.remove(), field_, CENTER_POINT, totalDelayed_, remainingDelayed_, pitch_);

    /* register new block */
    intervals_.clear();
    intervals_.add(current_block_);
    moveReceiver_.setBlock(current_block_);
    turnReceiver_.setBlock(current_block_);
    holdBlock_.setCurrentBlock(current_block_);
    shakeReceiver_.setBlock(current_block_);
    
    /* notify what create new block */
    for (BlockCreationListener listener : creationListeners_)
      listener.NotifyBlockCreation(current_block_);

    return new ThreadTask(new MoveAction
                          (Arrays.asList(new StackListener[] {field_}), gameListeners_, blockDownListeners_,
                           cutReceiver_, field_, current_block_, this, INIT_INTERVAL, INIT_WAIT_TIME, pitch_),
                          INIT_INTERVAL);
  }

  private int calculateScore(int lineNum)
  {
    int score = lineNum*lineNum*BASE_SCORE;
    try {
      for (Cell c : field_.getLineCells(GameSetting.FIELD_HEIGHT-1))
        if (c != null)
          throw new DummyThrowable();
      score *= BONUS_SCORE;
    }
    catch (DummyThrowable e) {}
    return score;
  }
   
  /* public methods */
  public void start(EventNotifier<MoveLineEventListener> moveLineCollector,
                    EventNotifier<RotateEventListener> rotateCollector,
                    EventNotifier<PitchListener> pitchCollector,
                    EventNotifier<CutCompleteListener> cutCollector,
                    EventNotifier<HoldEventListener> holdCollector,
                    EventNotifier<ShakeListener> shakeCollector,
                    EventNotifier<FallEventListener> fallCollector)   
  {
    moveLineCollector.addListener(moveReceiver_);
    rotateCollector.addListener(turnReceiver_);
    pitchCollector.addListener(pitch_);
    cutCollector.addListener(cutReceiver_);
    holdCollector.addListener(holdBlock_);
    shakeCollector.addListener(shakeReceiver_);
    
    processThread_ = new AutomaticProcessThread();
    try {
      processThread_.start(startNextAction(), pitch_,
                           totalDelayed_, remainingDelayed_,
                           intervals_);
    } catch (NotCreateBlock e) {
      System.out.println("the place where a object must not be throwed was throwed.");
      assert false;
      
    }
  }

  public void cutLines(Iterable<Integer> lines)
  {
    int lineNum = 0;
    for (@SuppressWarnings("unused") Integer tmp : lines)
      ++lineNum;
    
    score_ += calculateScore(lineNum);
    pitch_.updateRate(lineNum);
    shakePossibility_.update(lineNum);
  }
  
  public int getScore()
  {
    return score_;
  }

  public FieldOperator getField()
  {
    return field_;
  }

  public BlockOperator getCurrentBlock()
  {
    return current_block_;
  }

  public void addBlockCreationListener(BlockCreationListener listener)
  {
	  creationListeners_.add(listener);
  }
  
  public void addGameListener(GameListener listener)
  {
    gameListeners_.add(listener);
  }

  public Iterable<BlockType> getNextBlocks()
  {
    return next_blocks_;
  }
  
  public ThreadTask createNewBlock() throws NotCreateBlock
  {
    return startNextAction();
  }

  public void stop()
  {
    processThread_.interrupt();
    remainingTask_ = processThread_.getNextTask();
  }

  public void restart()
  {
    assert remainingTask_ != null;
    processThread_ = new AutomaticProcessThread();
    processThread_.start(remainingTask_, pitch_,
                         totalDelayed_, remainingDelayed_,
                         intervals_);
  }
  
  public void addBlockDownEventListner(BlockDownEventListener listener)
  {
	  blockDownListeners_.add(listener);
  }

  public boolean hold()
  {
    return holdBlock_.isHeld();
  }

  public int getShakeCount()
  {
    return shakePossibility_.getShakeCount();
  }
  public BlockOperator getHoldBlock()
  {
    return holdBlock_.getHoldBlock();
  }
  
  public void losed()
  {
    this.stop();
  }
}
