/*
 * Copyright (C) 2009 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.ac.osaka_u.ist.sel.android.BluetoothReversi;


import android.annotation.SuppressLint;
import android.app.ActionBar;
import android.app.Activity;
import android.app.AlertDialog;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.content.DialogInterface;
import android.content.Intent;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.util.Log;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.Button;
import android.widget.Toast;

/**
 * This is the main Activity that displays the current chat session.
 */
@SuppressLint({ "HandlerLeak", "InflateParams" })
public class BluetoothReversi extends Activity  {
	// Debugging
	private static final String TAG = "BluetoothReversi";
	private static final boolean D = true;

	// Message types sent from the BluetoothReversiService Handler
	public static final int MESSAGE_STATE_CHANGE = 1;
	public static final int MESSAGE_READ = 2;
	public static final int MESSAGE_WRITE = 3;
	public static final int MESSAGE_DEVICE_NAME = 4;
	public static final int MESSAGE_TOAST = 5;

	public static final int MESSAGE_CLIENT = 6;
	public static final int MESSAGE_SERVER = 7;

	// Key names received from the BluetoothReversiService Handler
	public static final String DEVICE_NAME = "device_name";
	public static final String TOAST = "toast";

	// Intent request codes
	private static final int REQUEST_CONNECT_DEVICE_SECURE = 1;
	private static final int REQUEST_CONNECT_DEVICE_INSECURE = 2;
	private static final int REQUEST_ENABLE_BT = 3;

	private static final int CLIENT = 0;
	private static final int SERVER = 1;

	private boolean recreateFlag = false;

	private Activity inner = this;

	// Layout Views
	private Button mStartButton_alone;
	private Button mStartButton_two;
	private Button mExitButton;
	private Button mQuitButton;
	private Button mRestartButton;

	private Button mBlankButton1;
	private Button mBlankButton2;

	// Name of the connected device
	private String mConnectedDeviceName = null;

	// Local Bluetooth adapter
	private BluetoothAdapter mBluetoothAdapter = null;

	// Member object for the chat services
	private BluetoothReversiService mReversiService = null;

	// BoardView object for setContentView()
	private BoardViewPair bv_pair;
	private BoardViewPairCircle bv_pair_circle;

	private final int REVERSI = 1;
	private final int NIP = 2;

	private int gameMode = REVERSI;


	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		if(D) Log.e(TAG, "+++ ON CREATE +++");

		// Set up the window layout
		setContentView(R.layout.main);

		// Get local Bluetooth adapter
		mBluetoothAdapter = BluetoothAdapter.getDefaultAdapter();

		// If the adapter is null, then Bluetooth is not supported
		if (mBluetoothAdapter == null) {
			Toast.makeText(this, "Bluetooth is not available", Toast.LENGTH_LONG).show();
			finish();
			return;
		}
	}

	@Override
	public void onStart() {
		super.onStart();
		if(D) Log.e(TAG, "++ ON START ++");

		// If BT is not on, request that it be enabled.
		// setupChat() will then be called during onActivityResult
		if (!mBluetoothAdapter.isEnabled()) {
			Intent enableIntent = new Intent(BluetoothAdapter.ACTION_REQUEST_ENABLE);
			startActivityForResult(enableIntent, REQUEST_ENABLE_BT);
			// Otherwise, setup the chat session
		} else {
			if (mReversiService == null) setupChat();
		}
	}

	@Override
	public synchronized void onResume() {
		super.onResume();
		if(D) Log.e(TAG, "+ ON RESUME +");

		// Performing this check in onResume() covers the case in which BT was
		// not enabled during onStart(), so we were paused to enable it...
		// onResume() will be called when ACTION_REQUEST_ENABLE activity returns.
		if (mReversiService != null) {
			// Only if the state is STATE_NONE, do we know that we haven't started already
			if (mReversiService.getState() == BluetoothReversiService.STATE_NONE) {
				// Start the Bluetooth chat services
				mReversiService.start();
			}
		}
	}

	public void setRecreateFlag(boolean f) {
		recreateFlag = f;
	}

	private void setupChat() {

		//blank button for layout
		mBlankButton1 = (Button) findViewById(R.id.button_blank1);
		mBlankButton2 = (Button) findViewById(R.id.button_blank2);

		mBlankButton1.setVisibility(Button.INVISIBLE);
		mBlankButton2.setVisibility(Button.INVISIBLE);

		// start alone button with a listener that for click events
		mStartButton_alone = (Button) findViewById(R.id.button_start_alone);
		mStartButton_alone.setOnClickListener(new OnClickListener() {

			public void onClick(View v) {

				if (mReversiService != null) {
					if (mReversiService.getState() == BluetoothReversiService.STATE_CONNECTED) {
						//pushed start alone button when connecting to other device,should stop()
						//need message?
						mReversiService.stop();
						return;
					}
				}

				//set a new layout for playing game
				setContentView(R.layout.game_alone);

				// restart button with a listener that for click events
				mRestartButton = (Button)findViewById(R.id.button_restart);
				mRestartButton.setOnClickListener(new OnClickListener() {
					public void onClick(View v) {

						BoardViewAlone bva = (BoardViewAlone)findViewById(R.id.gameView);
						if (bva.getThreadStat()) return;

						AlertDialog.Builder alertDlg = new AlertDialog.Builder(inner);
						alertDlg.setCancelable(false);
						alertDlg.setMessage("はじめからやりなおしますか？");
						alertDlg.setPositiveButton(
								"はい",
								new DialogInterface.OnClickListener() {
									public void onClick(DialogInterface dialog, int which) {
										// restart game
										BoardViewAlone bva = (BoardViewAlone)findViewById(R.id.gameView);
										bva.restartGame();
									}
								});
						alertDlg.setNegativeButton(
								"いいえ",
								new DialogInterface.OnClickListener() {
									public void onClick(DialogInterface dialog, int which) {
										//when pushed "No" button
									}
								});

						//show dialog
						alertDlg.create().show();
					}
				});


				//title button
				mQuitButton = (Button) findViewById(R.id.button_quit);
				mQuitButton.setOnClickListener(new OnClickListener() {

					@Override
					public void onClick(View v) {

						BoardViewAlone bva = (BoardViewAlone)findViewById(R.id.gameView);
						if (bva.getThreadStat()) return;

						//when pushed title button , make dialog
						AlertDialog.Builder alertDlg = new AlertDialog.Builder(inner);
						alertDlg.setCancelable(false);
						alertDlg.setMessage("タイトルに戻りますか？");
						alertDlg.setPositiveButton(
								"はい",
								new DialogInterface.OnClickListener() {
									public void onClick(DialogInterface dialog, int which) {
										//when pushed "Yes" button
										recreateFlag = true;
										recreate();

									}
								});
						alertDlg.setNegativeButton(
								"いいえ",
								new DialogInterface.OnClickListener() {
									public void onClick(DialogInterface dialog, int which) {
										//when pushed "No" button
									}
								});

						//show dialog
						alertDlg.create().show();
					}
				});
			}
		});

		// start two button with a listener that for click events
		mStartButton_two = (Button) findViewById(R.id.button_start_two);
		mStartButton_two.setOnClickListener(new OnClickListener() {
			public void onClick(View v) {
				// start game
				checkStartGame();
			}
		});

		// exit button with a listener that for click events
		mExitButton = (Button) findViewById(R.id.button_exit);
		mExitButton.setOnClickListener(new OnClickListener() {
			public void onClick(View v) {

				//make dialog
				AlertDialog.Builder alertDlg = new AlertDialog.Builder(inner);
				alertDlg.setCancelable(false);
				alertDlg.setMessage("アプリを終了しますか？");
				alertDlg.setPositiveButton(
						"はい",
						new DialogInterface.OnClickListener() {
							public void onClick(DialogInterface dialog, int which) {
								//when pushed "Yes" button
								finish();

							}
						});
				alertDlg.setNegativeButton(
						"いいえ",
						new DialogInterface.OnClickListener() {
							public void onClick(DialogInterface dialog, int which) {
								//when pushed "No" button

							}
						});

				//show dialog
				alertDlg.create().show();
			}
		});

		// Initialize the BluetoothChatService to perform bluetooth connections
		mReversiService = new BluetoothReversiService(this,mHandler);
	}

	@Override
	public synchronized void onPause() {
		super.onPause();
		if(D) Log.e(TAG, "- ON PAUSE -");
	}

	@Override
	public void onStop() {
		super.onStop();
		if(D) Log.e(TAG, "-- ON STOP --");
	}

	@Override
	public void onDestroy() {
		super.onDestroy();

		if (!recreateFlag) {
			Toast.makeText(this,"アプリケーションを終了しました", Toast.LENGTH_LONG).show();
		}
		recreateFlag = false;

		// Stop the Bluetooth Reversi services
		if (mReversiService != null) mReversiService.stop();
		if(D) Log.e(TAG, "--- ON DESTROY ---");
	}

	private void ensureDiscoverable() {
		if(D) Log.d(TAG, "ensure discoverable");
		if (mBluetoothAdapter.getScanMode() !=
				BluetoothAdapter.SCAN_MODE_CONNECTABLE_DISCOVERABLE) {
			Intent discoverableIntent = new Intent(BluetoothAdapter.ACTION_REQUEST_DISCOVERABLE);
			discoverableIntent.putExtra(BluetoothAdapter.EXTRA_DISCOVERABLE_DURATION, 300);
			startActivity(discoverableIntent);
		}
	}

	private void checkStartGame() {

		// Check that we're actually connected before trying anything
		if (mReversiService.getState() != BluetoothReversiService.STATE_CONNECTED) {
			Toast.makeText(this, R.string.not_connected, Toast.LENGTH_SHORT).show();
			return;
		}

		if (mReversiService.getPlayerStat() == SERVER) {

			//make dialog for deciding mode
			AlertDialog.Builder alertDlg = new AlertDialog.Builder(inner);
			alertDlg.setCancelable(false);
			alertDlg.setTitle("モード選択");
			alertDlg.setMessage("どちらのモードで遊びますか？");
			alertDlg.setPositiveButton(
					"ニップモード",
					new DialogInterface.OnClickListener() {
						public void onClick(DialogInterface dialog, int which) {
							//when pushed "Yes" button
							setContentView(R.layout.game_pair_circle);
							bv_pair_circle = (BoardViewPairCircle)findViewById(R.id.gameView_2p_circle);
							bv_pair_circle.setBluetoothReversiService(mReversiService);
							//write code the same as below!!!!!
							mQuitButton = (Button) findViewById(R.id.button_quit_2p_circle);
							gameMode = NIP;
							mQuitButton.setOnClickListener(new OnClickListener() {

								@Override
								public void onClick(View v) {

									if (bv_pair_circle.getThreadStat()) return ;


									//when pushed "go to title" button
									//make dialog
									AlertDialog.Builder alertDlg = new AlertDialog.Builder(inner);
									alertDlg.setCancelable(false);
									alertDlg.setMessage("タイトルに戻りますか？");
									alertDlg.setPositiveButton(
											"はい",
											new DialogInterface.OnClickListener() {
												public void onClick(DialogInterface dialog, int which) {
													//when pushed "Yes" button
													mReversiService.stop();
													recreateFlag = true;
													recreate();

												}
											});
									alertDlg.setNegativeButton(
											"いいえ",
											new DialogInterface.OnClickListener() {
												public void onClick(DialogInterface dialog, int which) {
													//when pushed "No" button

												}
											});

									//show dialog
									alertDlg.create().show();
								}
							});
						}
					});
			alertDlg.setNegativeButton(
					"リバーシモード",
					new DialogInterface.OnClickListener() {
						public void onClick(DialogInterface dialog, int which) {
							//when pushed "No" button
							setContentView(R.layout.game_pair);
							bv_pair = (BoardViewPair)findViewById(R.id.gameView_2p);
							bv_pair.setBluetoothReversiService(mReversiService);
							mQuitButton = (Button) findViewById(R.id.button_quit_2p);
							gameMode = REVERSI;
							mQuitButton.setOnClickListener(new OnClickListener() {

								@Override
								public void onClick(View v) {

									if (bv_pair.getThreadStat()) return ;

									//when pushed "go to title" button
									//make dialog
									AlertDialog.Builder alertDlg = new AlertDialog.Builder(inner);
									alertDlg.setCancelable(false);
									alertDlg.setMessage("タイトルに戻りますか？");
									alertDlg.setPositiveButton(
											"はい",
											new DialogInterface.OnClickListener() {
												public void onClick(DialogInterface dialog, int which) {
													//when pushed "Yes" button
													mReversiService.stop();
													recreateFlag = true;
													recreate();

												}
											});
									alertDlg.setNegativeButton(
											"いいえ",
											new DialogInterface.OnClickListener() {
												public void onClick(DialogInterface dialog, int which) {
													//when pushed "No" button

												}
											});

									//show dialog
									alertDlg.create().show();
								}
							});
						}
					});

			//show dialog
			alertDlg.create().show();

		} else {
			// if client ; wait message by toast
			Toast.makeText(this,"サーバ側がゲームを開始するまでお待ちください",Toast.LENGTH_SHORT).show();
		}
	}

	private final void setStatus(int resId) {
		final ActionBar actionBar = getActionBar();
		actionBar.setSubtitle(resId);
	}

	private final void setStatus(CharSequence subTitle) {
		final ActionBar actionBar = getActionBar();
		actionBar.setSubtitle(subTitle);
	}

	// The Handler that gets information back from the BluetoothChatService
	private final Handler mHandler = new Handler() {
		@Override
		public void handleMessage(Message msg) {
			switch (msg.what) {
			case MESSAGE_STATE_CHANGE:
				if(D) Log.i(TAG, "MESSAGE_STATE_CHANGE: " + msg.arg1);
				switch (msg.arg1) {
				case BluetoothReversiService.STATE_CONNECTED:

					setStatus(getString(R.string.title_connected_to, mConnectedDeviceName));
					break;
				case BluetoothReversiService.STATE_CONNECTING:
					setStatus(R.string.title_connecting);
					break;
				case BluetoothReversiService.STATE_LISTEN:
				case BluetoothReversiService.STATE_NONE:
					setStatus(R.string.title_not_connected);
					break;
				}
				break;
			case MESSAGE_WRITE:

				break;
			case MESSAGE_READ:
				byte[] readBuf = (byte[]) msg.obj;
				// construct a string from the valid bytes in the buffer
				String readMessage = new String(readBuf, 0, msg.arg1);

				if (readMessage.startsWith("sample")) {
					gameMode = REVERSI;
					setContentView(R.layout.game_pair);
					bv_pair = (BoardViewPair)findViewById(R.id.gameView_2p);
					mQuitButton = (Button) findViewById(R.id.button_quit_2p);
					mQuitButton.setOnClickListener(new OnClickListener() {

						@Override
						public void onClick(View v) {

							//when pushed "go to title" button
							//make dialog
							AlertDialog.Builder alertDlg = new AlertDialog.Builder(inner);
							alertDlg.setCancelable(false);
							alertDlg.setMessage("タイトルに戻りますか？");
							alertDlg.setPositiveButton(
									"Yes",
									new DialogInterface.OnClickListener() {
										public void onClick(DialogInterface dialog, int which) {
											//when pushed "Yes" button
											mReversiService.stop();
											recreateFlag = true;
											recreate();

										}
									});
							alertDlg.setNegativeButton(
									"No",
									new DialogInterface.OnClickListener() {
										public void onClick(DialogInterface dialog, int which) {
											//when pushed "No" button

										}
									});

							//show dialog
							alertDlg.create().show();
						}
					});

					bv_pair.setBluetoothReversiService(mReversiService,Integer.valueOf(readMessage.substring(6)));
					break;
				} else if (readMessage.startsWith("restart")) {

					if (gameMode == REVERSI) {
						bv_pair.restart(Integer.valueOf(readMessage.substring(7)));
					} else {
						bv_pair_circle.restart(Integer.valueOf(readMessage.substring(7)));
					}

					break;
				} else if(readMessage.startsWith("nip")){
					gameMode = NIP;
					setContentView(R.layout.game_pair_circle);
					bv_pair_circle = (BoardViewPairCircle)findViewById(R.id.gameView_2p_circle);
					mQuitButton = (Button) findViewById(R.id.button_quit_2p_circle);
					mQuitButton.setOnClickListener(new OnClickListener() {

						@Override
						public void onClick(View v) {

							//when pushed "go to title" button
							//make dialog
							AlertDialog.Builder alertDlg = new AlertDialog.Builder(inner);
							alertDlg.setCancelable(false);
							alertDlg.setMessage("タイトルに戻りますか？");
							alertDlg.setPositiveButton(
									"Yes",
									new DialogInterface.OnClickListener() {
										public void onClick(DialogInterface dialog, int which) {
											//when pushed "Yes" button
											mReversiService.stop();
											recreateFlag = true;
											recreate();

										}
									});
							alertDlg.setNegativeButton(
									"No",
									new DialogInterface.OnClickListener() {
										public void onClick(DialogInterface dialog, int which) {
											//when pushed "No" button

										}
									});

							//show dialog
							alertDlg.create().show();
						}
					});

					bv_pair_circle.setBluetoothReversiService(mReversiService,Integer.valueOf(readMessage.substring(3)));
				}else{

					if (gameMode == REVERSI) {
						bv_pair.setPlaeAfterRead(readMessage);
					} else {
						bv_pair_circle.setPlaeAfterRead(readMessage);
					}
				}
				break;
			case MESSAGE_DEVICE_NAME:
				// save the connected device's name
				mConnectedDeviceName = msg.getData().getString(DEVICE_NAME);
				Toast.makeText(getApplicationContext(), "Connected to "
						+ mConnectedDeviceName, Toast.LENGTH_SHORT).show();
				break;
			case MESSAGE_TOAST:
				Toast.makeText(getApplicationContext(), msg.getData().getString(TOAST),
						Toast.LENGTH_SHORT).show();
				break;
			}
		}
	};

	public void onActivityResult(int requestCode, int resultCode, Intent data) {
		if(D) Log.d(TAG, "onActivityResult " + resultCode);
		switch (requestCode) {
		case REQUEST_CONNECT_DEVICE_SECURE:
			// When DeviceListActivity returns with a device to connect
			if (resultCode == Activity.RESULT_OK) {
				connectDevice(data, true);
			}
			break;
		case REQUEST_CONNECT_DEVICE_INSECURE:
			// When DeviceListActivity returns with a device to connect
			if (resultCode == Activity.RESULT_OK) {
				connectDevice(data, false);
			}
			break;
		case REQUEST_ENABLE_BT:
			// When the request to enable Bluetooth returns
			if (resultCode == Activity.RESULT_OK) {
				// Bluetooth is now enabled, so set up a chat session
				setupChat();
			} else {
				// User did not enable Bluetooth or an error occurred
				Log.d(TAG, "BT not enabled");
				Toast.makeText(this, R.string.bt_not_enabled_leaving, Toast.LENGTH_SHORT).show();
				finish();
			}
		}
	}

	private void connectDevice(Intent data, boolean secure) {
		// Get the device MAC address
		String address = data.getExtras()
				.getString(DeviceListActivity.EXTRA_DEVICE_ADDRESS);
		// Get the BluetoothDevice object
		BluetoothDevice device = mBluetoothAdapter.getRemoteDevice(address);
		// Attempt to connect to the device
		mReversiService.connect(device, secure);
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		MenuInflater inflater = getMenuInflater();
		inflater.inflate(R.menu.option_menu, menu);
		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		Intent serverIntent = null;
		switch (item.getItemId()) {
		case R.id.secure_connect_scan:
			// Launch the DeviceListActivity to see devices and do scan
			serverIntent = new Intent(this, DeviceListActivity.class);
			startActivityForResult(serverIntent, REQUEST_CONNECT_DEVICE_SECURE);
			return true;
		case R.id.insecure_connect_scan:
			// Launch the DeviceListActivity to see devices and do scan
			serverIntent = new Intent(this, DeviceListActivity.class);
			startActivityForResult(serverIntent, REQUEST_CONNECT_DEVICE_INSECURE);
			return true;
		case R.id.discoverable:
			// Ensure this device is discoverable by others
			ensureDiscoverable();
			return true;
		}
		return false;
	}
}
