//PRET-Extractor
//Copyright (c) 2013 Tetsuya Kanda
//
//http://sel.ist.osaka-u.ac.jp/pret/
//
//Permission is hereby granted, free of charge, to any person obtaining
//a copy of this software and associated documentation files (the
//"Software"), to deal in the Software without restriction, including
//without limitation the rights to use, copy, modify, merge, publish,
//distribute, sublicense, and/or sell copies of the Software, and to
//permit persons to whom the Software is furnished to do so, subject to
//the following conditions:
//
//The above copyright notice and this permission notice shall be
//included in all copies or substantial portions of the Software.
//
//THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
//LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
//OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
//WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package jp.ac.osaka_u.ist.sel.pret.engine.diff;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

//Similarity(A, B) = |AB| / |AB|

/**
 * calc similarity using Cygwin diff
 * 
 * @author t-kanda
 * 
 */
public class CygwinDiff extends ADiffCmd {

	public final static int DIFF_LINE = 0;
	public final static int DIFF_TOKEN = 1;

	private final static char OLD_LINE = '-';
	private final static char NEW_LINE = '+';
	private final static char UNCHANGED_LINE = '*';

	/**
	 * diff options. ignore blank, blank line, in case.
	 */
	private final static String OPTION_BASE = "-iwB";

	/**
	 * diff options. unified
	 */
	// private final static String[] OPTIONS1 = { "-du" };
	private final static String[] OPTIONS1 = { OPTION_BASE, "-u" };

	/**
	 * diff options. changes to characters
	 */
	private final static String[] OPTIONS2 = { OPTION_BASE, "--old-line-format=\"" + OLD_LINE + "\"", "--new-line-format=\"" + NEW_LINE + "\"",
			"--unchanged-line-format=\"" + UNCHANGED_LINE + "\"" };
	private String diffCommand;

	public CygwinDiff(String command) {
		diffCommand = command;
	}

	private static String cygPath(String path) {
		return path.replaceAll("\\\\", "/").replace(":", "/cygdrive/");
	}

	@Override
	protected void calc() {
		String diff = execDiff(file1, file2, DIFF_TOKEN);
		int nUnionAB = diff.length() - 1; // remove \n
		int nIntersectionAB = 0;
		int nDel = 0;
		int nAdd = 0;
		for (char c : diff.toCharArray()) {
			switch (c) {
			case UNCHANGED_LINE:
				nIntersectionAB++;
				break;
			case OLD_LINE:
				nDel++;
				break;
			case NEW_LINE:
				nAdd++;
				break;
			}
		}
		sim = (float) nIntersectionAB / nUnionAB;
		add = nAdd;
		del = nDel;
		lcs = nIntersectionAB;
	}

	public String diff(final Path file1, final Path file2) {
		return execDiff(file1, file2, DIFF_LINE);
	}

	private String execDiff(final Path file1, final Path file2, int option) {
		final StringBuilder buffer = new StringBuilder();
		Process p = null;
		try {
			List<String> command = new ArrayList<>();
			command.add(diffCommand);
			// command.addAll(Arrays.asList(OPTIONS_BASE));
			switch (option) {
			case DIFF_TOKEN:
				command.addAll(Arrays.asList(OPTIONS2));
				break;
			case DIFF_LINE:
				command.addAll(Arrays.asList(OPTIONS1));
				break;
			default:
			}
			command.add(cygPath(file1.toAbsolutePath().toString()));
			command.add(cygPath(file2.toAbsolutePath().toString()));

			ProcessBuilder pb = new ProcessBuilder(command);
			p = pb.start();

			final InputStream is = p.getInputStream();
			final InputStream eis = p.getErrorStream();

			Thread outth = new Thread(new Runnable() {
				public void run() {
					try (BufferedReader br = new BufferedReader(new InputStreamReader(is))) {
						String result;
						while ((result = br.readLine()) != null) {
							buffer.append(result + "\n");
						}
					} catch (IOException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}
				}
			});

			Thread errth = new Thread(new Runnable() {
				public void run() {
					try (BufferedReader br = new BufferedReader(new InputStreamReader(eis))) {
						String result;
						while ((result = br.readLine()) != null) {
							System.err.print(file1 + " - " + file2 + ": ");
							System.err.println(result);
						}
					} catch (IOException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}
				}
			});

			outth.start();
			errth.start();
			p.waitFor();
			outth.join();
			errth.join();
			is.close();
			eis.close();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (InterruptedException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
			System.err.println("WHY");
			p.destroy();

		} finally {

		}
		return buffer.toString();
	}

}
