//PRET-Extractor
//Copyright (c) 2013 Tetsuya Kanda
//
//http://sel.ist.osaka-u.ac.jp/pret/
//
//Permission is hereby granted, free of charge, to any person obtaining
//a copy of this software and associated documentation files (the
//"Software"), to deal in the Software without restriction, including
//without limitation the rights to use, copy, modify, merge, publish,
//distribute, sublicense, and/or sell copies of the Software, and to
//permit persons to whom the Software is furnished to do so, subject to
//the following conditions:
//
//The above copyright notice and this permission notice shall be
//included in all copies or substantial portions of the Software.
//
//THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
//LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
//OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
//WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package jp.ac.osaka_u.ist.sel.pret.engine.data;

import java.io.Serializable;
import java.util.Iterator;
import java.util.Set;

import gnu.trove.iterator.TIntIterator;
import gnu.trove.list.TIntList;
import gnu.trove.list.array.TIntArrayList;
import gnu.trove.map.TIntIntMap;
import gnu.trove.map.TIntObjectMap;
import gnu.trove.map.hash.TIntIntHashMap;
import gnu.trove.map.hash.TIntObjectHashMap;
import gnu.trove.set.TIntSet;
import gnu.trove.set.hash.THashSet;
import gnu.trove.set.hash.TIntHashSet;

/**
 * undirected graph
 * 
 * @author t-kanda
 * 
 */
public class Group implements Serializable {

	private static final long serialVersionUID = 1922978534450625080L;
	private TIntSet files;
	private Set<Edge> edges;
	private TIntSet gatheredFiles;
	private TIntObjectMap<TIntList> gatheredFileMap;
	private Set<Edge> gatheredEdges;
	private char type;

	public Group(char type) {
		files = new TIntHashSet();
		edges = new THashSet<>();
		this.type = type;
	}

	public void addEdge(Edge edge) {
		edges.add(edge);
		files.add(edge.fileId1());
		files.add(edge.fileId2());
	}

	public boolean contains(int fileId) {
		return files.contains(fileId);
	}

	public void addAll(Group group) {
		this.edges.addAll(group.edges);
		this.files.addAll(group.files);
	}

	/**
	 * @return Set of file IDs
	 */
	public TIntSet files() {
		return files;
	}

	/**
	 * @return Set of edges
	 */
	public Set<Edge> edges() {
		return edges;
	}
	
	public char type(){
		return type;
	}

	/**
	 * @return Set of file IDs
	 */
	public TIntSet gatheredFiles() {
		return gatheredFiles;
	}

	public TIntList gathered(int gatheredId) {
		return gatheredFileMap.get(gatheredId);
	}

	public TIntObjectMap<TIntList> gatheredFileMap() {
		return gatheredFileMap;
	}

	/**
	 * @return returns edge set
	 */
	public Set<Edge> gatheredEdges() {
		return gatheredEdges;
	}

	public void gather(Group motherGroup, Set<Edge> newEdges) {
		this.gatheredFiles = motherGroup.gatheredFiles;
		this.gatheredFileMap = motherGroup.gatheredFileMap;
		this.gatheredEdges = newEdges;
	}

	public void gather(ISimilarity similarity) {

		gatheredFiles = new TIntHashSet(files);
		gatheredEdges = new THashSet<>(edges);
		gatheredFileMap = new TIntObjectHashMap<>();

		Set<TIntSet> sameSet = new THashSet<>();

		/*
		 * find edge which diff = 0
		 */
		for (Edge edge : edges) {
			// diff size is zero
			if (similarity.getDiffSize(edge) == 0) {

				gatheredEdges.remove(edge);

				TIntSet same1 = null, same2 = null;
				int fileId1 = edge.fileId1();
				int fileId2 = edge.fileId2();

				// already belongs to some group
				for (TIntSet same : sameSet) {
					if (same.contains(fileId1)) {
						same1 = same;
						if (same2 != null) {
							break;
						}
					}
					if (same.contains(fileId2)) {
						same2 = same;
						if (same1 != null) {
							break;
						}
					}
				}

				// if they are different group, merge.
				if (same1 != same2) {
					TIntSet n = new TIntHashSet();
					if (same1 != null) {
						n.addAll(same1);
						sameSet.remove(same1);
					} else {
						n.add(fileId1);
					}
					if (same2 != null) {
						n.addAll(same2);
						sameSet.remove(same2);
					} else {
						n.add(fileId2);
					}
					sameSet.add(n);
				} else if (same1 == null/* && same2 == null */) {
					TIntSet n = new TIntHashSet();
					n.add(fileId1);
					n.add(fileId2);
					sameSet.add(n);
				}
			}
		}

		/*
		 * attach id
		 */
		int correspondId = -1;
		TIntIntMap correspond = new TIntIntHashMap();

		for (TIntSet same : sameSet) {
			gatheredFiles.removeAll(same);
			gatheredFiles.add(correspondId);

			for (TIntIterator i = same.iterator(); i.hasNext();) {
				int fileId = i.next();
				correspond.put(fileId, correspondId);
			}
			TIntList sameList = new TIntArrayList(same);
			sameList.sort();

			gatheredFileMap.put(correspondId, sameList);
			correspondId--;
		}

		if (correspondId < -1) {
			/*
			 * change edge file - file(grouped) to edge file - group
			 */
			Set<Edge> newEdges = new THashSet<Edge>();
			for (Iterator<Edge> it = gatheredEdges.iterator(); it.hasNext();) {
				Edge edge = it.next();
				int fileId1 = edge.fileId1();
				int fileId2 = edge.fileId2();
				if (correspond.containsKey(fileId1)) {
					fileId1 = correspond.get(fileId1);
				}
				if (correspond.containsKey(fileId2)) {
					fileId2 = correspond.get(fileId2);
				}
				if (fileId1 < 0 || fileId2 < 0) {
					Edge newEdge = new Edge(fileId1, fileId2);
					it.remove();
					newEdges.add(newEdge);
				}
			}
			gatheredEdges.addAll(newEdges);
		}
	}

	public void noGather() {
		gatheredFiles = files;
		gatheredEdges = edges;
	}

	public int choose1(int fileId) {
		return fileId < 0 ? gathered(fileId).iterator().next() : fileId;
	}

}
