//PRET-Extractor
//Copyright (c) 2013 Tetsuya Kanda
//
//http://sel.ist.osaka-u.ac.jp/pret/
//
//Permission is hereby granted, free of charge, to any person obtaining
//a copy of this software and associated documentation files (the
//"Software"), to deal in the Software without restriction, including
//without limitation the rights to use, copy, modify, merge, publish,
//distribute, sublicense, and/or sell copies of the Software, and to
//permit persons to whom the Software is furnished to do so, subject to
//the following conditions:
//
//The above copyright notice and this permission notice shall be
//included in all copies or substantial portions of the Software.
//
//THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
//LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
//OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
//WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package jp.ac.osaka_u.ist.sel.pret.engine.data;

import gnu.trove.set.hash.THashSet;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Serializable;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import jp.ac.osaka_u.ist.sel.pret.util.EncodeDetector;

public class SimilarityOnDisk implements ISimilarity, Serializable {

	/**
	 * 
	 */
	private static final long serialVersionUID = -6487655544696948519L;

	/** many tmp files will be created... so group them into some subdirs */
	private static final int DIFFTMP_SIZE = 400;

	private URI tmp;
	private Set<Edge> edges;

	public SimilarityOnDisk(Path tmp) {
		Path simout = Paths.get(tmp.toString(), "sim");
		this.tmp = simout.toUri();
		try {
			Files.createDirectory(simout);
		} catch (IOException e) {
			// TODO Auto-generated catch block
			// e.printStackTrace();
		}
		edges = new THashSet<>();
	}

	@Override
	public void setSimilarity(Edge edge, float similarity) {

		synchronized (edges) {
			edges.add(edge);
		}

		Path p = diffPath(edge);
		Path dir = p.getParent();
		if (!Files.exists(dir)) {
			try {
				Files.createDirectory(dir);
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
		try (BufferedWriter bw = Files.newBufferedWriter(p, Charset.defaultCharset())) {
			bw.append(Float.toString(similarity));
			bw.append('\n');
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	@Override
	public float getSimilarity(Edge edge) {
		if (edges.contains(edge)) {
			Path p = diffPath(edge);
			if (Files.exists(p)) {
				try (BufferedReader fin = Files.newBufferedReader(p, EncodeDetector.charset(p))) {
					String line = fin.readLine(); // sim
					if (line != null) {
						return Float.parseFloat(line);
					}
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
		return NOVALUE;
	}

	@Override
	public void setDiff(Edge edge, String diff) {
		Path p = diffPath(edge);
		Path dir = p.getParent();
		if (!Files.exists(dir)) {
			try {
				Files.createDirectory(dir);
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
		try (BufferedWriter bw = Files.newBufferedWriter(p, Charset.defaultCharset(), StandardOpenOption.WRITE)) {
			String diffText = diff.replaceAll("\r\n", "\n").replaceAll("[^<>].*\\n", "");
			// System.out.println(diffText);
			int dec = 0;
			for (char c : diffText.toCharArray()) {
				if (c == '<') {
					dec++;
				}
			}
			int inc = diffText.length() - dec;

			bw.append(String.format("dec %d\n inc %d\ndiff %d\ndiffTXT\n", dec, inc, diffText.length()));
			bw.append(diff);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	@Override
	public String getDiff(Edge edge) {
		if (edges.contains(edge)) {
			Path p = diffPath(edge);
			if (Files.exists(p)) {
				StringBuilder result = new StringBuilder();
				String line;
				try (BufferedReader fin = Files.newBufferedReader(p, EncodeDetector.charset(p))) {
					while (null != (line = fin.readLine())) {
						if(line.equals("diffTXT")){
							break;
						}
					}
					while (null != (line = fin.readLine())) {
						result.append(line);
						result.append('\n');
					}
					return result.toString();
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
		return null;
	}

	@Override
	public int getDiffSize(Edge edge) {
		if (edges.contains(edge)) {
			Path p = diffPath(edge);
			if (Files.exists(p)) {
				try {
					for (String s : Files.readAllLines(p, EncodeDetector.charset(p))) {
						String[] sp = s.split(" ");
						if (sp[0].equals("diff")) {
							return Integer.parseInt(sp[1]);
						}
					}
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
		return NODIFF;
	}

	@Override
	public int getDiffSize(int v, int u) {
		return getDiffSize(new Edge(v, u));
	}

	@Override
	public int getDec(Edge edge) {
		if (edges.contains(edge)) {
			Path p = diffPath(edge);
			if (Files.exists(p)) {
				try {
					for (String s : Files.readAllLines(p, EncodeDetector.charset(p))) {
						String[] sp = s.split(" ");
						if (sp[0].equals("dec")) {
							return Integer.parseInt(sp[1]);
						}
					}
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
		return NODIFF;
	}

	@Override
	public int getInc(Edge edge) {
		if (edges.contains(edge)) {
			Path p = diffPath(edge);
			if (Files.exists(p)) {
				try {
					for (String s : Files.readAllLines(p, EncodeDetector.charset(p))) {
						String[] sp = s.split(" ");
						if (sp[0].equals("inc")) {
							return Integer.parseInt(sp[1]);
						}
					}
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
		return NODIFF;
	}

	@Override
	public void setIncDec(Edge edge, int add, int del) {
		Path p = diffPath(edge);
		Path dir = p.getParent();
		if (!Files.exists(dir)) {
			try {
				Files.createDirectory(dir);
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}

		List<String> buf = new ArrayList<>();

		boolean binc = false, bdec = false, bdiff = false, bread = false;
		if (Files.exists(p)) {
			try {
				for (String s : Files.readAllLines(p, EncodeDetector.charset(p))) {
					if (!bread) {
						String[] sp = s.split(" ");
						switch (sp[0]) {
						case "inc":
							buf.add("inc " + add);
							binc = true;
							break;
						case "dec":
							buf.add("dec " + del);
							bdec = true;
							break;
						case "diff":
							buf.add("diff " + (add + del));
							bdiff = true;
							break;
						case "diffTXT":
							bread = true;
						default:
							buf.add(s);
						}
					} else {
						buf.add(s);
					}
				}
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
		if (!binc) {
			buf.add("inc " + add);
		}
		if (!bdec) {
			buf.add("dec " + del);
		}
		if (!bdiff) {
			buf.add("diff " + (add + del));
		}

		try {
			Files.write(p, buf, Charset.defaultCharset());
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 
	 * @param edge
	 *            target edge
	 * @return subdir number
	 */
	private int difftmpSmall(Edge edge) {
		return edge.hashCode() / DIFFTMP_SIZE;
	}

	private Path diffPath(Edge edge) {
		String fileName = new StringBuilder().append(edge.fileId1()).append('_').append(edge.fileId2()).append(".diff").toString();
		return Paths.get(Paths.get(tmp).toString(), Integer.toString(difftmpSmall(edge)), fileName);
	}

	@Override
	public Set<Edge> edges() {
		return new THashSet<>(edges);
	}
}
