//PRET-Extractor
//Copyright (c) 2013 Tetsuya Kanda
//
//http://sel.ist.osaka-u.ac.jp/pret/
//
//Permission is hereby granted, free of charge, to any person obtaining
//a copy of this software and associated documentation files (the
//"Software"), to deal in the Software without restriction, including
//without limitation the rights to use, copy, modify, merge, publish,
//distribute, sublicense, and/or sell copies of the Software, and to
//permit persons to whom the Software is furnished to do so, subject to
//the following conditions:
//
//The above copyright notice and this permission notice shall be
//included in all copies or substantial portions of the Software.
//
//THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
//LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
//OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
//WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package jp.ac.osaka_u.ist.sel.pret.engine.data;

import gnu.trove.map.TObjectFloatMap;
import gnu.trove.map.TObjectIntMap;
import gnu.trove.map.hash.TObjectFloatHashMap;
import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.hash.THashSet;

import java.io.Serializable;
import java.util.Set;

/**
 * 
 * keep similarity and length of diff on memory. does not save diff text.
 * 
 * @author t-kanda
 * 
 */
public class SimilarityOnMemorySkipDiff implements ISimilarity, Serializable {

	/**
	 * 
	 */
	private static final long serialVersionUID = 3004214404223078368L;
	
	private TObjectFloatMap<Edge> sim;
	private TObjectIntMap<Edge> incm, decm;

	public SimilarityOnMemorySkipDiff() {
		sim = new TObjectFloatHashMap<>();
		incm = new TObjectIntHashMap<>();
		decm = new TObjectIntHashMap<>();
	}

	@Override
	public void setSimilarity(Edge edge, float similarity) {
		synchronized (sim) {
			sim.put(edge, similarity);
		}
	}

	@Override
	public float getSimilarity(Edge edge) {
		if (sim.containsKey(edge)) {
			return sim.get(edge);
		}
		return NOVALUE;
	}

	@Override
	public void setDiff(Edge edge, String diff) {

		String diffText = diff.replaceAll("\r\n", "\n").replaceAll("[^<>].*\\n", "");
		// System.out.println(diffText);
		int dec = 0;
		for (char c : diffText.toCharArray()) {
			if (c == '<') {
				dec++;
			}
		}
		int inc = diffText.length() - dec;
		synchronized (incm) {
			incm.put(edge, inc);
		}
		synchronized (decm) {
			decm.put(edge, dec);
		}
	}

	@Override
	public String getDiff(Edge edge) {
		if (incm.containsKey(edge)) {
			return "";
		}
		return null;
	}

	@Override
	public int getDiffSize(Edge edge) {
		if (sim.containsKey(edge)) {
			return incm.get(edge) + decm.get(edge);
		}
		return NODIFF;
	}

	@Override
	public int getDiffSize(int v, int u) {
		return getDiffSize(new Edge(v, u));
	}

	@Override
	public int getDec(Edge edge) {
		if (decm.containsKey(edge)) {
			return decm.get(edge);
		}
		return ISimilarity.NODIFF;
	}

	@Override
	public int getInc(Edge edge) {
		if (incm.containsKey(edge)) {
			return incm.get(edge);
		}
		return ISimilarity.NODIFF;
	}

	@Override
	public Set<Edge> edges() {
		return new THashSet<>(sim.keySet());
	}

	@Override
	public void setIncDec(Edge edge,int add, int del) {
		synchronized (decm) {
			decm.put(edge, del);
		}
		synchronized (incm) {
			incm.put(edge, add);
		}
	}

}
