//PRET-Extractor
//Copyright (c) 2013 Tetsuya Kanda
//
//http://sel.ist.osaka-u.ac.jp/pret/
//
//Permission is hereby granted, free of charge, to any person obtaining
//a copy of this software and associated documentation files (the
//"Software"), to deal in the Software without restriction, including
//without limitation the rights to use, copy, modify, merge, publish,
//distribute, sublicense, and/or sell copies of the Software, and to
//permit persons to whom the Software is furnished to do so, subject to
//the following conditions:
//
//The above copyright notice and this permission notice shall be
//included in all copies or substantial portions of the Software.
//
//THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
//LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
//OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
//WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package jp.ac.osaka_u.ist.sel.pret.engine.graph;

import gnu.trove.TIntCollection;
import gnu.trove.iterator.TIntIterator;

import java.io.BufferedWriter;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

import jp.ac.osaka_u.ist.sel.pret.engine.Target;
import jp.ac.osaka_u.ist.sel.pret.engine.data.DirectedEdge;
import jp.ac.osaka_u.ist.sel.pret.engine.data.DirectedGroup;
import jp.ac.osaka_u.ist.sel.pret.engine.data.Edge;
import jp.ac.osaka_u.ist.sel.pret.engine.data.FileInfo;
import jp.ac.osaka_u.ist.sel.pret.engine.data.ISimilarity;

/**
 * output dot file
 * 
 * @author t-kanda
 * 
 */
public class Dot {

	public final static String COLOR_G = "#FFFFFF";

	/**
	 * set line width
	 * 
	 * @param diffSize
	 * @return
	 */
	private static float linewidth(int diffSize) {
		// return (float) Math.abs(diffSize * 0.01 + 0.3);
		return 2.0f;
	}

	/**
	 * make label string for multiple vertex
	 * 
	 * @param gathered
	 * @param files
	 * @return
	 */
	private static String fileNameSet(TIntCollection gathered, Target target) {
		StringBuilder buffer = new StringBuilder();
		for (TIntIterator it = gathered.iterator(); it.hasNext();) {
			int fid = it.next();
			FileInfo fi = target.getFile(fid);
			buffer.append(fi.projectId());
			buffer.append(':');
			buffer.append(fi.fileName());
			buffer.append('(');
			buffer.append(fid);
			buffer.append(')');
			buffer.append("\\n");
		}
		return buffer.toString();
	}

	public static Path dot(Path output, int groupId, DirectedGroup group, Target target, ISimilarity diffmap) {
		Path tmp = Paths.get(output.toString(), groupId + ".dot");
		try (BufferedWriter bw = Files.newBufferedWriter(tmp, Charset.defaultCharset())) {
			graph(groupId, group, target, diffmap, bw);
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return tmp;
	}

	public static void graph(int groupId, DirectedGroup group, Target target, ISimilarity diffmap, BufferedWriter bw) throws IOException {
		bw.append("digraph GROUP_" + groupId + "{ \n");
		bw.append("overlap = scale;\n");

		/*
		 * set vertex
		 */
		for (TIntIterator i = group.gatheredFiles().iterator(); i.hasNext();) {
			int node = i.next();

			String label;
			if (node < 0) {
				label = fileNameSet(group.gathered(node), target);
			} else {
				StringBuilder buffer = new StringBuilder();
				FileInfo fi = target.getFile(node);
				buffer.append(fi.projectId());
				buffer.append(':');
				buffer.append(fi.fileName().replaceAll("\\\\", "\\\\\\\\"));
				buffer.append('(');
				buffer.append(node);
				buffer.append(')');
				label = buffer.toString();
			}

			bw.append(String.format("%6d [style=filled, shape = box, fillcolor = \"%s\", label=\"%s\"];\n", node, COLOR_G, label));
		}

		/*
		 * set edge
		 */
		for (DirectedEdge edge : group.gatheredEdges()) {
			int from = edge.from();
			int to = edge.to();
			Edge sample = new Edge(group.choose1(from), group.choose1(to));
			int diffSize = diffmap.getDiffSize(sample);
			String direction = edge.hasDirection() ? "" : "dir=none";
			bw.append(String.format("%d -> %d [label = \"%d\" penwidth=%.1f %s];\n", from, to, diffSize, linewidth(diffSize), direction));
		}

		bw.append("}");
	}
}
