//PRET-Extractor
//Copyright (c) 2013 Tetsuya Kanda
//
//http://sel.ist.osaka-u.ac.jp/pret/
//
//Permission is hereby granted, free of charge, to any person obtaining
//a copy of this software and associated documentation files (the
//"Software"), to deal in the Software without restriction, including
//without limitation the rights to use, copy, modify, merge, publish,
//distribute, sublicense, and/or sell copies of the Software, and to
//permit persons to whom the Software is furnished to do so, subject to
//the following conditions:
//
//The above copyright notice and this permission notice shall be
//included in all copies or substantial portions of the Software.
//
//THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
//LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
//OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
//WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package jp.ac.osaka_u.ist.sel.pret.engine.data;

import gnu.trove.map.TObjectFloatMap;
import gnu.trove.map.TObjectIntMap;
import gnu.trove.map.hash.TObjectFloatHashMap;
import gnu.trove.map.hash.TObjectIntHashMap;
import gnu.trove.set.hash.THashSet;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Serializable;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Set;

import jp.ac.osaka_u.ist.sel.pret.util.EncodeDetector;

/**
 * 
 * keep similarity on memory, save diff text on disk
 * 
 * @author t-kanda
 * 
 */
public class SimilarityOnMemoryDiffOnDisk implements ISimilarity, Serializable {
	/**
	 * 
	 */
	private static final long serialVersionUID = -9022741990492017710L;

	/** many tmp files will be created... so group them into some subdirs */
	private static final int DIFFTMP_SIZE = 400;

	private URI tmp;
	private TObjectFloatMap<Edge> sim;
	private TObjectIntMap<Edge> incm, decm;

	public SimilarityOnMemoryDiffOnDisk(Path tmp) {
		Path simout = Paths.get(tmp.toString(), "sim");
		this.tmp = simout.toUri();
		try {
			Files.createDirectory(simout);
		} catch (IOException e) {
			// TODO Auto-generated catch block
			// e.printStackTrace();
		}
		sim = new TObjectFloatHashMap<>();
		incm = new TObjectIntHashMap<>();
		decm = new TObjectIntHashMap<>();
	}

	@Override
	public void setSimilarity(Edge edge, float similarity) {
		synchronized (sim) {
			sim.put(edge, similarity);
		}
	}

	@Override
	public float getSimilarity(Edge edge) {
		if (sim.containsKey(edge)) {
			return sim.get(edge);
		}
		return NOVALUE;
	}

	@Override
	public void setDiff(Edge edge, String diff) {

		String diffText = diff.replaceAll("\r\n", "\n").replaceAll("[^<>].*\\n", "");
		// System.out.println(diffText);
		int dec = 0;
		for (char c : diffText.toCharArray()) {
			if (c == '<') {
				dec++;
			}
		}
		int inc = diffText.length() - dec;
		synchronized (incm) {
			incm.put(edge, inc);
		}
		synchronized (decm) {
			decm.put(edge, dec);
		}

		Path p = diffPath(edge);
		Path dir = p.getParent();
		if (!Files.exists(dir)) {
			try {
				Files.createDirectory(dir);
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}

		try (BufferedWriter bw = Files.newBufferedWriter(p, Charset.defaultCharset())) {
			bw.append(diff);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	@Override
	public String getDiff(Edge edge) {
		if (sim.containsKey(edge)) {
			Path p = diffPath(edge);
			if (Files.exists(p)) {
				StringBuilder result = new StringBuilder();
				String line;
				try (BufferedReader fin = Files.newBufferedReader(p, EncodeDetector.charset(p))) {
					while (null != (line = fin.readLine())) {
						result.append(line);
						result.append('\n');
					}
					return result.toString();
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
		return null;
	}

	@Override
	public int getDiffSize(Edge edge) {
		// return di.getDiffSize(edge);
		if (sim.containsKey(edge)) {
			return incm.get(edge) + decm.get(edge);
		}
		return NODIFF;
	}

	@Override
	public int getDiffSize(int v, int u) {
		return getDiffSize(new Edge(v, u));
	}

	@Override
	public int getDec(Edge edge) {
		if (decm.containsKey(edge)) {
			return decm.get(edge);
		}
		return ISimilarity.NODIFF;
	}

	@Override
	public int getInc(Edge edge) {
		if (incm.containsKey(edge)) {
			return incm.get(edge);
		}
		return ISimilarity.NODIFF;
	}

	/**
	 * 
	 * @param edge
	 *            target edge
	 * @return subdir number
	 */
	private int difftmpSmall(Edge edge) {
		return edge.hashCode() / DIFFTMP_SIZE;
	}

	private Path diffPath(Edge edge) {
		String fileName = new StringBuilder().append(edge.fileId1()).append('_').append(edge.fileId2()).append(".diff").toString();
		return Paths.get(Paths.get(tmp).toString(), Integer.toString(difftmpSmall(edge)), fileName);
	}

	@Override
	public Set<Edge> edges() {
		return new THashSet<>(sim.keySet());
	}

	@Override
	public void setIncDec(Edge edge,int add, int del) {
		synchronized (decm) {
			decm.put(edge, del);
		}
		synchronized (incm) {
			incm.put(edge, add);
		}
	}
}
