//PRET-Extractor
//Copyright (c) 2013 Tetsuya Kanda
//
//http://sel.ist.osaka-u.ac.jp/pret/
//
//Permission is hereby granted, free of charge, to any person obtaining
//a copy of this software and associated documentation files (the
//"Software"), to deal in the Software without restriction, including
//without limitation the rights to use, copy, modify, merge, publish,
//distribute, sublicense, and/or sell copies of the Software, and to
//permit persons to whom the Software is furnished to do so, subject to
//the following conditions:
//
//The above copyright notice and this permission notice shall be
//included in all copies or substantial portions of the Software.
//
//THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
//EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
//MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
//NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
//LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
//OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
//WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

package jp.ac.osaka_u.ist.sel.pret.engine.graph;

import gnu.trove.iterator.TIntIterator;
import gnu.trove.set.TIntSet;
import gnu.trove.set.hash.TIntHashSet;
import jp.ac.osaka_u.ist.sel.pret.engine.data.Edge;
import jp.ac.osaka_u.ist.sel.pret.engine.data.Group;
import jp.ac.osaka_u.ist.sel.pret.engine.data.ISimilarity;

/**
 * Minimum spanning tree (Prim)
 * 
 * @author t-kanda
 * 
 */
public class SpanningTree {

	/**
	 * Minimum spanning tree (Prim)
	 * 
	 * @param group
	 *            target group
	 * @return tree (target group with limited edges)
	 */
	public static Group getSpanningTree(Group group, ISimilarity similarity, boolean gather) {

		if (group.gatheredFiles().size() == 1) {
			return group;
		}

		// U : vertexes not added in the tree
		TIntSet unknown = new TIntHashSet(group.gatheredFiles());
		// any vertex can be a start point
		int start = unknown.iterator().next();
		int maxSize = unknown.size();
		unknown.remove(start);

		// V : vertexes in the tree
		TIntSet vNew = new TIntHashSet((int) (maxSize * 1.25));
		vNew.add(start);

		Group sp = new Group(group.type());
		// continue until all vertexes are included in the tree
		while (vNew.size() != maxSize) {
			int newNode = -1;
			int minValue = Integer.MAX_VALUE;
			Edge minEdge = null;
			// find edge from v in V to...@
			for (TIntIterator i = vNew.iterator(); i.hasNext();) {
				int vg = i.next();
				int v = group.choose1(vg);
				// to u in U
				for (TIntIterator j = unknown.iterator(); j.hasNext();) {
					int ug = j.next();
					int u = group.choose1(ug);
					Edge gatheredEdge = new Edge(vg, ug);
					int diffSize = similarity.getDiffSize(v, u);
					if (diffSize != ISimilarity.NODIFF && diffSize < minValue) {
						minValue = diffSize;
						minEdge = gatheredEdge;
						newNode = ug;
					}
				}
			}

			// add edge with minimum cost to the tree
			sp.addEdge(minEdge);
			vNew.add(newNode);
			unknown.remove(newNode);
		}

		if (gather) {
			sp.gather(group, sp.edges());
		} else {
			sp.noGather();
		}
		return sp;
	}
}
